<?php

namespace App\Http\Controllers\Front;

use Illuminate\{
    Http\Request,
};

use App\{
    Models\Item,
    Models\Category,
    Http\Controllers\Controller,
};
use App\Helpers\PriceHelper;
use App\Models\Attribute;
use App\Models\AttributeOption;
use App\Models\Brand;
use App\Models\ChieldCategory;
use App\Models\Order;
use App\Models\Setting;
use App\Models\Subcategory;
use Illuminate\Support\Facades\Session;

class CatalogController extends Controller
{
    public function __construct()
    {
        $this->middleware('localize');
    }

	public function index(Request $request)
{
    // Fetch settings
    $setting = Setting::first();
    
    // Attribute filtering
    $attr_item_ids = [];
    if ($request->attribute) {
        $attr_item_ids = Attribute::where('name', $request->attribute)->pluck('item_id')->toArray();
    }

    // Attribute options filtering
    $option_attr_ids = [];
    if ($request->option) {
        $option_attr_ids = AttributeOption::whereIn('name', explode(',', $request->option))->pluck('attribute_id')->toArray();
    }

    // Get items matching selected attribute options
    $option_wise_item_ids = Attribute::whereIn('id', $option_attr_ids)->pluck('item_id')->toArray();

    // Get filters
    $sorting = $request->sorting ?? null;
    $new = $request->has('new') ? 1 : null;
    $feature = $request->quick_filter == 'feature' ? 1 : null;
    $top = $request->quick_filter == 'top' ? 1 : null;
    $best = $request->quick_filter == 'best' ? 1 : null;
    $new = $request->quick_filter == 'new' ? 1 : null;
    $brand = $request->brand ? Brand::whereSlug($request->brand)->first() : null;
    $search = $request->search ?? null;
    $category = $request->category ? Category::whereSlug($request->category)->first() : null;
    $subcategory = $request->subcategory ? Subcategory::whereSlug($request->subcategory)->first() : null;
    $childcategory = $request->childcategory ? ChieldCategory::whereSlug($request->childcategory)->first() : null;
    $minPrice = $request->minPrice ? PriceHelper::convertPrice($request->minPrice) : null;
    $maxPrice = $request->maxPrice ? PriceHelper::convertPrice($request->maxPrice) : null;
    $tag = $request->tag ?? null;

    // Query products
    $items = Item::with('category')
        ->when($category, fn($query, $category) => $query->where('category_id', $category->id))
        ->when($subcategory, fn($query, $subcategory) => $query->where('subcategory_id', $subcategory->id))
        ->when($childcategory, fn($query, $childcategory) => $query->where('childcategory_id', $childcategory->id))
        ->when($feature, fn($query) => $query->whereIsType('feature'))
        ->when($tag, fn($query, $tag) => $query->where('tags', 'like', '%' . $tag . '%'))
        ->when($new, fn($query) => $query->orderBy('id', 'desc'))
        ->when($top, fn($query) => $query->whereIsType('top'))
        ->when($best, fn($query) => $query->whereIsType('best'))
        ->when($brand, fn($query, $brand) => $query->where('brand_id', $brand->id))
        ->when($search, function ($query, $search) {
            return $query->where('status', 1)->where(function ($q) use ($search) {
                $q->where('name', 'like', "%$search%")->orWhere('tags', 'like', "%$search%");
            });
        })
        ->when($minPrice, fn($query, $minPrice) => $query->where('discount_price', '>=', $minPrice))
        ->when($maxPrice, fn($query, $maxPrice) => $query->where('discount_price', '<=', $maxPrice))
        ->when($sorting, function ($query, $sorting) {
            return $query->orderBy('discount_price', $sorting == 'low_to_high' ? 'asc' : 'desc');
        })
        ->when($attr_item_ids, fn($query, $attr_item_ids) => $query->whereIn('id', $attr_item_ids))
        ->when($option_wise_item_ids, fn($query, $option_wise_item_ids) => $query->whereIn('id', $option_wise_item_ids))
        ->where('status', 1)
        ->orderBy('id', 'desc')
        ->paginate($setting->view_product);

    // Calculate total sales count
    $totalSales = 0;
    foreach ($items as $item) {
        $totalSales += $item->sales_count;
    }

    // Get attributes and options
    $attributeIds = AttributeOption::groupBy('name')->pluck('attribute_id')->unique();
    $attributes = Attribute::withCount('options')->whereIn('id', $attributeIds)->get();

    // View Type
    if ($request->view_check) {
        Session::put('view_catalog', $request->view_check);
    }

    $checkType = Session::get('view_catalog', 'grid');
    $name_string_count = $checkType === 'grid' ? 38 : 55;

    return view($request->ajax() ? 'front.catalog.catalog' : 'front.catalog.index', [
        'attributes' => $attributes,
        'brand' => $brand,
        'items' => $items,
        'totalSales' => $totalSales,
        'name_string_count' => $name_string_count,
        'category' => $category,
        'subcategory' => $subcategory,
        'childcategory' => $childcategory,
        'checkType' => $checkType,
        'brands' => Brand::withCount('items')->whereStatus(1)->get(),
        'categories' => Category::whereStatus(1)->orderBy('serial', 'asc')->withCount(['items' => fn($query) => $query->where('status', 1)])->get(),
    ]);
}

    public function getProductSalesCount($product_slug)
    {
        $orders = Order::all(); // Fetch all orders
        $totalSales = 0;
    
        foreach ($orders as $order) {
            $cart = json_decode($order->cart, true); // Decode cart JSON
    
            foreach ($cart as $item) {
                if ($item['slug'] === $product_slug) {
                    $totalSales += intval($item['qty']);
                }
            }
        }
    
        return $totalSales;
    }

    public function showProductSales($slug)
    {
        $salesCount = $this->getProductSalesCount($slug);
        return view('product-sales', compact('salesCount', 'slug'));
    }


    public function viewType($type)
    {
        Session::put('view_catalog',$type);
        return response()->json($type);
    }


    public function suggestSearch(Request $request)
    {
        $category = null;
        if($request->category){
            $category = Category::whereSlug($request->category)->first();
        }
        $search = $request->search;
        $items = Item::whereStatus(1)
        ->when($search, function ($query, $search) {
            return $query->where('name', 'like', '%' . $search . '%')->orderby('id','desc')->take(10);
        })
        ->when($category, function ($query, $category) {
            return $query->where('category_id', $category->id);
        })
        ->get();

        return view('includes.search_suggest',compact('items'));
    }

}
